"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group"
import { Label } from "@/components/ui/label"
import { Check, Loader2 } from "lucide-react"

type StatusUpdateFormProps = {
  registrationId: string
  currentStatus: string
  updateStatus: (id: string, status: string) => Promise<{ success: boolean; message: string }>
}

export function StatusUpdateForm({ registrationId, currentStatus, updateStatus }: StatusUpdateFormProps) {
  const [status, setStatus] = useState(currentStatus)
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [message, setMessage] = useState<{ type: "success" | "error"; text: string } | null>(null)

  async function handleSubmit() {
    setIsSubmitting(true)
    setMessage(null)

    try {
      const result = await updateStatus(registrationId, status)

      if (result.success) {
        setMessage({ type: "success", text: result.message })
      } else {
        setMessage({ type: "error", text: result.message })
      }
    } catch (error) {
      setMessage({ type: "error", text: "Tilan päivitys epäonnistui" })
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <div className="space-y-4">
      <RadioGroup value={status} onValueChange={setStatus}>
        <div className="flex items-center space-x-2">
          <RadioGroupItem value="pending" id="pending" />
          <Label htmlFor="pending">Odottaa käsittelyä</Label>
        </div>
        <div className="flex items-center space-x-2">
          <RadioGroupItem value="processed" id="processed" />
          <Label htmlFor="processed">Käsitelty</Label>
        </div>
        <div className="flex items-center space-x-2">
          <RadioGroupItem value="accepted" id="accepted" />
          <Label htmlFor="accepted">Hyväksytty</Label>
        </div>
        <div className="flex items-center space-x-2">
          <RadioGroupItem value="rejected" id="rejected" />
          <Label htmlFor="rejected">Hylätty</Label>
        </div>
      </RadioGroup>

      {message && (
        <div className={`text-sm ${message.type === "success" ? "text-green-600" : "text-red-600"}`}>
          {message.text}
        </div>
      )}

      <Button onClick={handleSubmit} disabled={isSubmitting || status === currentStatus} className="w-full">
        {isSubmitting ? (
          <>
            <Loader2 className="mr-2 h-4 w-4 animate-spin" />
            Päivitetään...
          </>
        ) : (
          <>
            <Check className="mr-2 h-4 w-4" />
            Päivitä tila
          </>
        )}
      </Button>
    </div>
  )
}
