"use client"

import { useState } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import * as z from "zod"
import { Button } from "@/components/ui/button"
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group"
import { Checkbox } from "@/components/ui/checkbox"
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert"
import { CheckCircle2, AlertCircle } from "lucide-react"
import dataService from "@/app/lib/data-service"

const formSchema = z.object({
  childName: z.string().min(2, { message: "Lapsen nimi on pakollinen" }),
  childBirthdate: z.string().min(1, { message: "Syntymäaika on pakollinen" }),
  guardianName: z.string().min(2, { message: "Huoltajan nimi on pakollinen" }),
  email: z.string().email({ message: "Virheellinen sähköpostiosoite" }),
  phone: z.string().min(5, { message: "Puhelinnumero on pakollinen" }),
  address: z.string().min(5, { message: "Osoite on pakollinen" }),
  startDate: z.string().min(1, { message: "Aloituspäivä on pakollinen" }),
  careType: z.enum(["full", "part"], { required_error: "Valitse hoitomuoto" }),
  specialDiet: z.string().optional(),
  allergies: z.string().optional(),
  additionalInfo: z.string().optional(),
  termsAccepted: z.literal(true, {
    errorMap: () => ({ message: "Sinun täytyy hyväksyä ehdot jatkaaksesi" }),
  }),
})

type FormValues = z.infer<typeof formSchema>

export default function RegistrationPage() {
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitStatus, setSubmitStatus] = useState<{ success: boolean; message: string } | null>(null)

  const form = useForm<FormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      childName: "",
      childBirthdate: "",
      guardianName: "",
      email: "",
      phone: "",
      address: "",
      startDate: "",
      careType: "full",
      specialDiet: "",
      allergies: "",
      additionalInfo: "",
      termsAccepted: false,
    },
  })

  async function onSubmit(data: FormValues) {
    try {
      setIsSubmitting(true)
      setSubmitStatus(null)

      // Save registration using our data service
      dataService.saveRegistration(data)

      setSubmitStatus({
        success: true,
        message: "Rekisteröinti onnistui! Otamme sinuun yhteyttä pian.",
      })

      if (form) {
        form.reset()
      }
    } catch (error) {
      console.error("Registration error:", error)
      setSubmitStatus({
        success: false,
        message: "Rekisteröinti epäonnistui. Yritä myöhemmin uudelleen.",
      })
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <div className="container mx-auto py-10 px-4">
      <h1 className="text-3xl font-bold mb-6">Rekisteröi lapsesi</h1>
      <p className="text-gray-600 mb-8">
        Täytä alla oleva lomake rekisteröidäksesi lapsesi Satakieli päiväkotiin. Otamme sinuun yhteyttä mahdollisimman
        pian.
      </p>

      {submitStatus && (
        <Alert
          className={`mb-6 ${
            submitStatus.success
              ? "bg-green-50 text-green-800 border-green-200"
              : "bg-red-50 text-red-800 border-red-200"
          }`}
        >
          {submitStatus.success ? (
            <CheckCircle2 className="h-4 w-4 text-green-600" />
          ) : (
            <AlertCircle className="h-4 w-4 text-red-600" />
          )}
          <AlertTitle>{submitStatus.success ? "Rekisteröinti onnistui!" : "Virhe!"}</AlertTitle>
          <AlertDescription>{submitStatus.message}</AlertDescription>
        </Alert>
      )}

      <div className="bg-white p-6 rounded-lg shadow-md">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <div className="grid md:grid-cols-2 gap-6">
              <div className="space-y-6">
                <h2 className="text-xl font-semibold border-b pb-2">Lapsen tiedot</h2>

                <FormField
                  control={form.control}
                  name="childName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Lapsen nimi *</FormLabel>
                      <FormControl>
                        <Input placeholder="Etunimi Sukunimi" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="childBirthdate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Syntymäaika *</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <h2 className="text-xl font-semibold border-b pb-2 mt-8">Huoltajan tiedot</h2>

                <FormField
                  control={form.control}
                  name="guardianName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Huoltajan nimi *</FormLabel>
                      <FormControl>
                        <Input placeholder="Etunimi Sukunimi" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="email"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Sähköposti *</FormLabel>
                      <FormControl>
                        <Input type="email" placeholder="esimerkki@email.com" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="phone"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Puhelinnumero *</FormLabel>
                      <FormControl>
                        <Input placeholder="+358 50 123 4567" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="address"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Osoite *</FormLabel>
                      <FormControl>
                        <Input placeholder="Katuosoite, Postinumero, Kaupunki" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="space-y-6">
                <h2 className="text-xl font-semibold border-b pb-2">Hoitotiedot</h2>

                <FormField
                  control={form.control}
                  name="startDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Toivottu aloituspäivä *</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="careType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Hoitomuoto *</FormLabel>
                      <FormControl>
                        <RadioGroup
                          onValueChange={field.onChange}
                          defaultValue={field.value}
                          className="flex flex-col space-y-1"
                        >
                          <FormItem className="flex items-center space-x-3 space-y-0">
                            <FormControl>
                              <RadioGroupItem value="full" />
                            </FormControl>
                            <FormLabel className="font-normal">Kokopäivähoito</FormLabel>
                          </FormItem>
                          <FormItem className="flex items-center space-x-3 space-y-0">
                            <FormControl>
                              <RadioGroupItem value="part" />
                            </FormControl>
                            <FormLabel className="font-normal">Osapäivähoito</FormLabel>
                          </FormItem>
                        </RadioGroup>
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="specialDiet"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Erityisruokavalio</FormLabel>
                      <FormControl>
                        <Textarea placeholder="Kerro mahdollisista erityisruokavalioista" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="allergies"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Allergiat</FormLabel>
                      <FormControl>
                        <Textarea placeholder="Kerro mahdollisista allergioista" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="additionalInfo"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Lisätietoja</FormLabel>
                      <FormControl>
                        <Textarea placeholder="Muuta huomioitavaa, esim. lapsen erityistarpeet" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="termsAccepted"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0 mt-6">
                      <FormControl>
                        <Checkbox checked={field.value} onCheckedChange={field.onChange} />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>
                          Hyväksyn{" "}
                          <a href="/tietosuojaseloste" className="text-blue-600 hover:underline">
                            tietosuojaselosteen
                          </a>{" "}
                          ja annan luvan tietojeni käsittelyyn *
                        </FormLabel>
                        <FormDescription>Tietojasi käytetään vain rekisteröinnin käsittelyyn</FormDescription>
                        <FormMessage />
                      </div>
                    </FormItem>
                  )}
                />
              </div>
            </div>

            <div className="flex justify-end">
              <Button type="submit" disabled={isSubmitting} className="bg-rainbow-blue hover:bg-rainbow-blue/90">
                {isSubmitting ? "Lähetetään..." : "Lähetä rekisteröinti"}
              </Button>
            </div>
          </form>
        </Form>
      </div>
    </div>
  )
}
