"use client"

import { useEffect, useState } from "react"
import Link from "next/link"
import Image from "next/image"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import dataService, { type SiteContent } from "@/app/lib/data-service"

export default function Home() {
  const [content, setContent] = useState<SiteContent | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // Initialize sample data if needed
    dataService.initializeWithSampleData()

    // Load content
    const data = dataService.getSiteContent()
    setContent(data)
    setIsLoading(false)
  }, [])

  if (isLoading || !content) {
    return <div className="container mx-auto py-10 px-4">Loading...</div>
  }

  return (
    <div className="flex flex-col min-h-screen">
      {/* Hero Section */}
      <section className="relative py-20 md:py-32 overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-r from-rainbow-blue/20 to-rainbow-violet/20 -z-10" />
        <div className="container grid grid-cols-1 md:grid-cols-2 gap-8 items-center">
          <div className="space-y-6">
            <h1 className="text-4xl md:text-5xl font-bold tracking-tight">{content.hero.title}</h1>
            <p className="text-xl text-muted-foreground">{content.hero.subtitle}</p>
            <div className="flex flex-col sm:flex-row gap-4">
              <Button asChild size="lg" className="bg-rainbow-blue hover:bg-rainbow-blue/90">
                <Link href="/rekisterointi">Rekisteröi lapsesi</Link>
              </Button>
              <Button asChild size="lg" variant="outline">
                <Link href="/tietoa-meista">Lue lisää</Link>
              </Button>
            </div>
          </div>
          <div className="relative h-[300px] md:h-[400px] rounded-lg overflow-hidden shadow-xl">
            <Image src="/images/children-circle.jpeg" alt="Lapset leikkimässä" fill className="object-cover" priority />
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section className="py-16 bg-muted/30">
        <div className="container">
          <h2 className="text-3xl font-bold text-center mb-12">Miksi valita meidät?</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <Card className="border-rainbow-red/20 hover:border-rainbow-red/50 transition-colors">
              <CardContent className="pt-6">
                <div className="rounded-full bg-rainbow-red/10 w-12 h-12 flex items-center justify-center mb-4">
                  <svg
                    xmlns="http://www.w3.org/2000/svg"
                    fill="none"
                    viewBox="0 0 24 24"
                    strokeWidth={1.5}
                    stroke="currentColor"
                    className="w-6 h-6 text-rainbow-red"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      d="M15.182 15.182a4.5 4.5 0 0 1-6.364 0M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0ZM9.75 9.75c0 .414-.168.75-.375.75S9 10.164 9 9.75 9.168 9 9.375 9s.375.336.375.75Zm-.375 0h.008v.015h-.008V9.75Zm5.625 0c0 .414-.168.75-.375.75s-.375-.336-.375-.75.168-.75.375-.75.375.336.375.75Zm-.375 0h.008v.015h-.008V9.75Z"
                    />
                  </svg>
                </div>
                <h3 className="text-xl font-semibold mb-2">Ammattitaitoinen henkilökunta</h3>
                <p className="text-muted-foreground">
                  Kokenut ja koulutettu henkilökuntamme tarjoaa laadukasta varhaiskasvatusta.
                </p>
              </CardContent>
            </Card>

            <Card className="border-rainbow-yellow/20 hover:border-rainbow-yellow/50 transition-colors">
              <CardContent className="pt-6">
                <div className="rounded-full bg-rainbow-yellow/10 w-12 h-12 flex items-center justify-center mb-4">
                  <svg
                    xmlns="http://www.w3.org/2000/svg"
                    fill="none"
                    viewBox="0 0 24 24"
                    strokeWidth={1.5}
                    stroke="currentColor"
                    className="w-6 h-6 text-rainbow-yellow"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      d="M12 18v-5.25m0 0a6.01 6.01 0 0 0 1.5-.189m-1.5.189a6.01 6.01 0 0 1-1.5-.189m3.75 7.478a12.06 12.06 0 0 1-4.5 0m3.75 2.383a14.406 14.406 0 0 1-3 0M14.25 18v-.192c0-.983.658-1.823 1.508-2.316a7.5 7.5 0 1 0-7.517 0c.85.493 1.509 1.333 1.509 2.316V18"
                    />
                  </svg>
                </div>
                <h3 className="text-xl font-semibold mb-2">Monipuolinen toiminta</h3>
                <p className="text-muted-foreground">
                  Tarjoamme monipuolista toimintaa, joka tukee lapsen kehitystä ja oppimista.
                </p>
              </CardContent>
            </Card>

            <Card className="border-rainbow-green/20 hover:border-rainbow-green/50 transition-colors">
              <CardContent className="pt-6">
                <div className="rounded-full bg-rainbow-green/10 w-12 h-12 flex items-center justify-center mb-4">
                  <svg
                    xmlns="http://www.w3.org/2000/svg"
                    fill="none"
                    viewBox="0 0 24 24"
                    strokeWidth={1.5}
                    stroke="currentColor"
                    className="w-6 h-6 text-rainbow-green"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      d="M9 12.75 11.25 15 15 9.75M21 12c0 1.268-.63 2.39-1.593 3.068a3.745 3.745 0 0 1-1.043 3.296 3.745 3.745 0 0 1-3.296 1.043A3.745 3.745 0 0 1 12 21c-1.268 0-2.39-.63-3.068-1.593a3.746 3.746 0 0 1-3.296-1.043 3.745 3.745 0 0 1-1.043-3.296A3.745 3.745 0 0 1 3 12c0-1.268.63-2.39 1.593-3.068a3.745 3.745 0 0 1 1.043-3.296 3.746 3.746 0 0 1 3.296-1.043A3.746 3.746 0 0 1 12 3c1.268 0 2.39.63 3.068 1.593a3.746 3.746 0 0 1 3.296 1.043 3.746 3.746 0 0 1 1.043 3.296A3.745 3.745 0 0 1 21 12Z"
                    />
                  </svg>
                </div>
                <h3 className="text-xl font-semibold mb-2">Turvallinen ympäristö</h3>
                <p className="text-muted-foreground">
                  Luomme turvallisen ja viihtyisän ympäristön, jossa lapsi voi kasvaa ja kehittyä.
                </p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Activities Section */}
      <section className="py-16">
        <div className="container">
          <h2 className="text-3xl font-bold text-center mb-12">Toimintamme</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            <div className="relative h-[300px] rounded-lg overflow-hidden shadow-lg">
              <Image
                src="/images/children-sandbox.jpeg"
                alt="Lapset leikkimässä hiekkalaatikossa"
                width={500}
                height={300}
                className="object-cover w-full h-full"
              />
            </div>
            <div className="space-y-4 flex flex-col justify-center">
              <h3 className="text-2xl font-semibold">Luova toiminta</h3>
              <p className="text-muted-foreground">
                Tarjoamme monipuolista luovaa toimintaa, kuten askartelua, musiikkia ja taidetta. Luova toiminta
                kehittää lapsen mielikuvitusta, hienomotoriikkaa ja itseilmaisua.
              </p>
              <Button asChild variant="outline" className="self-start">
                <Link href="/palvelut">Lue lisää toiminnastamme</Link>
              </Button>
            </div>

            <div className="space-y-4 flex flex-col justify-center md:order-3">
              <h3 className="text-2xl font-semibold">Ulkoilu ja liikunta</h3>
              <p className="text-muted-foreground">
                Päivittäinen ulkoilu ja liikunta ovat tärkeä osa toimintaamme. Liikunta kehittää lapsen motorisia
                taitoja, koordinaatiota ja tasapainoa.
              </p>
              <Button asChild variant="outline" className="self-start">
                <Link href="/palvelut">Tutustu liikuntamahdollisuuksiin</Link>
              </Button>
            </div>
            <div className="relative h-[300px] rounded-lg overflow-hidden shadow-lg md:order-4">
              <Image
                src="/images/child-playground.jpeg"
                alt="Lapsi leikkimässä leikkipuistossa"
                width={500}
                height={300}
                className="object-cover w-full h-full"
              />
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-16 bg-gradient-to-r from-rainbow-indigo/20 to-rainbow-violet/20">
        <div className="container text-center space-y-6">
          <h2 className="text-3xl font-bold">Kiinnostuitko?</h2>
          <p className="text-xl text-muted-foreground max-w-2xl mx-auto">
            Rekisteröi lapsesi nyt ja anna hänelle mahdollisuus kasvaa ja kehittyä turvallisessa ja virikkeellisessä
            ympäristössä.
          </p>
          <Button asChild size="lg" className="mt-4 bg-rainbow-violet hover:bg-rainbow-violet/90">
            <Link href="/rekisterointi">Rekisteröi lapsesi nyt</Link>
          </Button>
        </div>
      </section>
    </div>
  )
}
