"use client"

// Types for our data models
export type Registration = {
  id: string
  childName: string
  childBirthdate: string
  guardianName: string
  email: string
  phone: string
  address: string
  startDate: string
  careType: string
  specialDiet?: string
  allergies?: string
  additionalInfo?: string
  termsAccepted: boolean
  status: string
  createdAt: string
}

export type SiteContent = {
  hero: {
    title: string
    subtitle: string
    description: string
  }
  about: {
    mainText: string
    mission: string
    vision: string
  }
  services: {
    mainText: string
    servicesList: string[]
  }
  contact: {
    address: string
    phone: string
    email: string
    hours: string
    mapUrl: string
  }
}

// Default content if none exists
const defaultContent: SiteContent = {
  hero: {
    title: "Tervetuloa Satakieli päiväkotiin",
    subtitle: "Turvallista ja laadukasta varhaiskasvatusta Turussa",
    description:
      "Satakieli päiväkoti tarjoaa laadukasta varhaiskasvatusta 1-6 -vuotiaille lapsille viihtyisässä ja turvallisessa ympäristössä. Toimintamme perustuu lapsilähtöisyyteen ja yksilölliseen huomioimiseen.",
  },
  about: {
    mainText:
      "Satakieli päiväkoti on perustettu vuonna 2010 tarjoamaan laadukasta varhaiskasvatusta 1-6 -vuotiaille lapsille. Päiväkotimme sijaitsee viihtyisässä ympäristössä Turun keskustan läheisyydessä, hyvien kulkuyhteyksien varrella.",
    mission:
      "Tarjoamme laadukasta varhaiskasvatusta turvallisessa ja virikkeellisessä ympäristössä. Tuemme lapsen kasvua, kehitystä ja oppimista yhteistyössä perheiden kanssa.",
    vision:
      "Haluamme olla Turun johtava päiväkoti, jossa lapset oppivat ja kasvavat onnellisina. Tavoitteenamme on luoda pohja elinikäiselle oppimiselle ja hyvinvoinnille.",
  },
  services: {
    mainText: "Tarjoamme monipuolisia varhaiskasvatuspalveluita 1-6 -vuotiaille lapsille.",
    servicesList: ["Kokopäivähoito", "Osapäivähoito", "Esiopetus", "Erityisopetus", "Kielikylpy"],
  },
  contact: {
    address: "Brahenkatu 20, 20100 Turku",
    phone: "+358 40 7790659",
    email: "brahensatakieli@gmail.com",
    hours: "Ma-Pe: 7:00 - 17:00, La-Su: Suljettu",
    mapUrl: "https://maps.app.goo.gl/AryPJz4FZGSfddM69?g_st=aw",
  },
}

// Data service class
class DataService {
  // Get all registrations
  getRegistrations(): Registration[] {
    if (typeof window === "undefined") return []

    const data = localStorage.getItem("registrations")
    if (!data) return []

    try {
      return JSON.parse(data)
    } catch (e) {
      console.error("Error parsing registrations:", e)
      return []
    }
  }

  // Get a single registration by ID
  getRegistrationById(id: string): Registration | null {
    const registrations = this.getRegistrations()
    return registrations.find((r) => r.id === id) || null
  }

  // Save a new registration
  saveRegistration(registration: Omit<Registration, "id" | "status" | "createdAt">): Registration {
    const registrations = this.getRegistrations()

    const newRegistration: Registration = {
      ...registration,
      id: Date.now().toString(),
      status: "pending",
      createdAt: new Date().toISOString(),
    }

    registrations.push(newRegistration)
    localStorage.setItem("registrations", JSON.stringify(registrations))

    return newRegistration
  }

  // Update a registration
  updateRegistration(id: string, data: Partial<Registration>): Registration | null {
    const registrations = this.getRegistrations()
    const index = registrations.findIndex((r) => r.id === id)

    if (index === -1) return null

    const updatedRegistration = {
      ...registrations[index],
      ...data,
    }

    registrations[index] = updatedRegistration
    localStorage.setItem("registrations", JSON.stringify(registrations))

    return updatedRegistration
  }

  // Get site content
  getSiteContent(): SiteContent {
    if (typeof window === "undefined") return defaultContent

    const data = localStorage.getItem("siteContent")
    if (!data) {
      // Initialize with default content if none exists
      localStorage.setItem("siteContent", JSON.stringify(defaultContent))
      return defaultContent
    }

    try {
      return JSON.parse(data)
    } catch (e) {
      console.error("Error parsing site content:", e)
      return defaultContent
    }
  }

  // Update site content
  updateSiteContent(content: Partial<SiteContent>): SiteContent {
    const currentContent = this.getSiteContent()

    const updatedContent = {
      ...currentContent,
      ...content,
      // Handle nested updates
      hero: content.hero ? { ...currentContent.hero, ...content.hero } : currentContent.hero,
      about: content.about ? { ...currentContent.about, ...content.about } : currentContent.about,
      services: content.services ? { ...currentContent.services, ...content.services } : currentContent.services,
      contact: content.contact ? { ...currentContent.contact, ...content.contact } : currentContent.contact,
    }

    localStorage.setItem("siteContent", JSON.stringify(updatedContent))
    return updatedContent
  }

  // Initialize with sample data if needed
  initializeWithSampleData() {
    // Only initialize if no data exists
    if (this.getRegistrations().length === 0) {
      const sampleRegistrations: Registration[] = [
        {
          id: "1",
          childName: "Matti Meikäläinen",
          childBirthdate: "2019-05-15",
          guardianName: "Liisa Meikäläinen",
          email: "liisa.meikalainen@example.com",
          phone: "+358 50 123 4567",
          address: "Esimerkkikatu 1, 20100 Turku",
          startDate: "2023-08-15",
          careType: "full",
          specialDiet: "Laktoositon",
          allergies: "Pähkinät",
          additionalInfo: "Pitää erityisesti ulkoilusta ja piirtämisestä.",
          termsAccepted: true,
          status: "pending",
          createdAt: "2023-05-10T10:30:00.000Z",
        },
        {
          id: "2",
          childName: "Anna Virtanen",
          childBirthdate: "2020-02-20",
          guardianName: "Mikko Virtanen",
          email: "mikko.virtanen@example.com",
          phone: "+358 40 987 6543",
          address: "Puistokatu 5, 20100 Turku",
          startDate: "2023-09-01",
          careType: "part",
          specialDiet: "",
          allergies: "",
          additionalInfo: "Ujo uusissa tilanteissa.",
          termsAccepted: true,
          status: "accepted",
          createdAt: "2023-06-05T14:15:00.000Z",
        },
      ]

      localStorage.setItem("registrations", JSON.stringify(sampleRegistrations))
    }

    // Initialize content if it doesn't exist
    if (!localStorage.getItem("siteContent")) {
      localStorage.setItem("siteContent", JSON.stringify(defaultContent))
    }
  }
}

// Create and export a singleton instance
const dataService = new DataService()
export default dataService
