"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Textarea } from "@/components/ui/textarea"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Check, Loader2 } from "lucide-react"
import Image from "next/image"

export default function SettingsPage() {
  const [activeTab, setActiveTab] = useState("general")
  const [isSaving, setIsSaving] = useState(false)
  const [saveMessage, setSaveMessage] = useState<string | null>(null)

  // Mock settings data
  const [settings, setSettings] = useState({
    general: {
      siteName: "Satakieli Päiväkoti",
      siteDescription: "Kieliä tiedostava päiväkoti Turussa",
      logo: "/images/logo-blue.jpeg",
    },
    contact: {
      address: "Brahenkatu 20, 20100 Turku",
      phone: "+358 40 7790659",
      email: "brahensatakieli@gmail.com",
      hours: "Ma-Pe: 7:00 - 17:00, La-Su: Suljettu",
      mapUrl: "https://maps.app.goo.gl/AryPJz4FZGSfddM69?g_st=aw",
    },
    social: {
      facebook: "https://facebook.com/satakieli",
      instagram: "https://instagram.com/satakieli",
    },
  })

  const handleSave = () => {
    setIsSaving(true)
    setSaveMessage(null)

    // Simulate API call
    setTimeout(() => {
      setIsSaving(false)
      setSaveMessage("Asetukset tallennettu onnistuneesti!")

      // Clear message after 3 seconds
      setTimeout(() => {
        setSaveMessage(null)
      }, 3000)
    }, 1000)
  }

  const handleChange = (section: keyof typeof settings, field: string, value: string) => {
    setSettings({
      ...settings,
      [section]: {
        ...settings[section],
        [field]: value,
      },
    })
  }

  return (
    <div className="p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">Asetukset</h1>
        <Button onClick={handleSave} disabled={isSaving}>
          {isSaving ? (
            <>
              <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              Tallennetaan...
            </>
          ) : (
            <>
              <Check className="mr-2 h-4 w-4" />
              Tallenna muutokset
            </>
          )}
        </Button>
      </div>

      {saveMessage && (
        <div className="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">{saveMessage}</div>
      )}

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="mb-6">
          <TabsTrigger value="general">Yleiset</TabsTrigger>
          <TabsTrigger value="contact">Yhteystiedot</TabsTrigger>
          <TabsTrigger value="social">Sosiaalinen media</TabsTrigger>
        </TabsList>

        <TabsContent value="general">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Sivuston tiedot</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="siteName">Sivuston nimi</Label>
                  <Input
                    id="siteName"
                    value={settings.general.siteName}
                    onChange={(e) => handleChange("general", "siteName", e.target.value)}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="siteDescription">Kuvaus</Label>
                  <Textarea
                    id="siteDescription"
                    value={settings.general.siteDescription}
                    onChange={(e) => handleChange("general", "siteDescription", e.target.value)}
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Logo</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex justify-center">
                  <div className="relative w-32 h-32 rounded-lg overflow-hidden border">
                    <Image
                      src={settings.general.logo || "/placeholder.svg"}
                      alt="Logo"
                      fill
                      className="object-contain"
                    />
                  </div>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="logo">Vaihda logo</Label>
                  <div className="flex gap-2">
                    <Input id="logo" type="file" accept="image/*" />
                    <Button variant="outline" size="icon">
                      <Check className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
                <div className="pt-2">
                  <p className="text-sm text-muted-foreground">
                    Suositeltu koko: 200x200 pikseliä, PNG tai JPEG-muodossa.
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="contact">
          <Card>
            <CardHeader>
              <CardTitle>Yhteystiedot</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="address">Osoite</Label>
                <Input
                  id="address"
                  value={settings.contact.address}
                  onChange={(e) => handleChange("contact", "address", e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="phone">Puhelinnumero</Label>
                <Input
                  id="phone"
                  value={settings.contact.phone}
                  onChange={(e) => handleChange("contact", "phone", e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="email">Sähköposti</Label>
                <Input
                  id="email"
                  value={settings.contact.email}
                  onChange={(e) => handleChange("contact", "email", e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="hours">Aukioloajat</Label>
                <Input
                  id="hours"
                  value={settings.contact.hours}
                  onChange={(e) => handleChange("contact", "hours", e.target.value)}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="mapUrl">Google Maps -linkki</Label>
                <Input
                  id="mapUrl"
                  value={settings.contact.mapUrl}
                  onChange={(e) => handleChange("contact", "mapUrl", e.target.value)}
                />
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="social">
          <Card>
            <CardHeader>
              <CardTitle>Sosiaalisen median linkit</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="facebook">Facebook</Label>
                <Input
                  id="facebook"
                  value={settings.social.facebook}
                  onChange={(e) => handleChange("social", "facebook", e.target.value)}
                  placeholder="https://facebook.com/sivusi"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="instagram">Instagram</Label>
                <Input
                  id="instagram"
                  value={settings.social.instagram}
                  onChange={(e) => handleChange("social", "instagram", e.target.value)}
                  placeholder="https://instagram.com/sivusi"
                />
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}
