"use client"

import { useEffect, useState } from "react"
import Link from "next/link"
import { Button } from "@/components/ui/button"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Badge } from "@/components/ui/badge"
import { Eye } from "lucide-react"
import { ClientAdminLayout } from "@/components/admin/client-layout"
import dataService, { type Registration } from "@/app/lib/data-service"

export default function RegistrationsPage() {
  const [registrations, setRegistrations] = useState<Registration[]>([])
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // Initialize sample data if needed
    dataService.initializeWithSampleData()

    // Load registrations
    const data = dataService.getRegistrations()
    setRegistrations(data)
    setIsLoading(false)
  }, [])

  if (isLoading) {
    return (
      <ClientAdminLayout>
        <div className="p-6">Loading...</div>
      </ClientAdminLayout>
    )
  }

  return (
    <ClientAdminLayout>
      <div className="p-6">
        <div className="flex justify-between items-center mb-6">
          <h1 className="text-2xl font-bold">Rekisteröinnit</h1>
        </div>

        <div className="bg-white rounded-lg shadow overflow-hidden">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Lapsen nimi</TableHead>
                <TableHead>Huoltaja</TableHead>
                <TableHead>Yhteystiedot</TableHead>
                <TableHead>Aloituspäivä</TableHead>
                <TableHead>Hoitomuoto</TableHead>
                <TableHead>Tila</TableHead>
                <TableHead>Toiminnot</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {registrations.length > 0 ? (
                registrations.map((registration) => (
                  <TableRow key={registration.id}>
                    <TableCell className="font-medium">{registration.childName}</TableCell>
                    <TableCell>{registration.guardianName}</TableCell>
                    <TableCell>
                      <div>{registration.email}</div>
                      <div className="text-sm text-muted-foreground">{registration.phone}</div>
                    </TableCell>
                    <TableCell>{new Date(registration.startDate).toLocaleDateString("fi-FI")}</TableCell>
                    <TableCell>{registration.careType === "full" ? "Kokopäivähoito" : "Osapäivähoito"}</TableCell>
                    <TableCell>
                      <Badge
                        variant={registration.status === "pending" ? "outline" : "default"}
                        className={
                          registration.status === "pending" ? "bg-yellow-100 text-yellow-800 hover:bg-yellow-100" : ""
                        }
                      >
                        {registration.status === "pending" ? "Odottaa" : "Käsitelty"}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <Link href={`/admin/registrations/${registration.id}`}>
                        <Button variant="ghost" size="sm">
                          <Eye className="h-4 w-4 mr-1" />
                          Näytä
                        </Button>
                      </Link>
                    </TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={7} className="text-center py-4 text-muted-foreground">
                    Ei rekisteröintejä vielä.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>
      </div>
    </ClientAdminLayout>
  )
}
