"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Upload, Trash2 } from "lucide-react"
import Image from "next/image"

// Mock images for demonstration
const mockImages = [
  { id: "1", name: "children-circle.jpeg", url: "/images/children-circle.jpeg", type: "image/jpeg", size: "245 KB" },
  { id: "2", name: "children-sandbox.jpeg", url: "/images/children-sandbox.jpeg", type: "image/jpeg", size: "198 KB" },
  { id: "3", name: "child-playground.jpeg", url: "/images/child-playground.jpeg", type: "image/jpeg", size: "176 KB" },
  { id: "4", name: "children-ballpit.jpeg", url: "/images/children-ballpit.jpeg", type: "image/jpeg", size: "220 KB" },
  { id: "5", name: "logo-blue.jpeg", url: "/images/logo-blue.jpeg", type: "image/jpeg", size: "32 KB" },
  { id: "6", name: "logo-yellow.jpeg", url: "/images/logo-yellow.jpeg", type: "image/jpeg", size: "28 KB" },
  { id: "7", name: "logo-pink.jpeg", url: "/images/logo-pink.jpeg", type: "image/jpeg", size: "30 KB" },
]

export default function MediaPage() {
  const [activeTab, setActiveTab] = useState("all")
  const [selectedImage, setSelectedImage] = useState<string | null>(null)
  const [isUploading, setIsUploading] = useState(false)

  // Filter images based on active tab
  const filteredImages =
    activeTab === "all"
      ? mockImages
      : activeTab === "logos"
        ? mockImages.filter((img) => img.name.includes("logo"))
        : mockImages.filter((img) => !img.name.includes("logo"))

  const handleUpload = (e: React.FormEvent) => {
    e.preventDefault()
    setIsUploading(true)

    // Simulate upload delay
    setTimeout(() => {
      setIsUploading(false)
      // In a real app, you would handle the file upload here
    }, 1500)
  }

  return (
    <div className="p-6">
      <h1 className="text-2xl font-bold mb-6">Media</h1>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2">
          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <div className="flex justify-between items-center mb-4">
              <TabsList>
                <TabsTrigger value="all">Kaikki</TabsTrigger>
                <TabsTrigger value="photos">Kuvat</TabsTrigger>
                <TabsTrigger value="logos">Logot</TabsTrigger>
              </TabsList>
            </div>

            <TabsContent value={activeTab} className="mt-0">
              <div className="bg-white p-4 rounded-lg shadow">
                <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-4">
                  {filteredImages.map((image) => (
                    <Card
                      key={image.id}
                      className={`overflow-hidden cursor-pointer transition-all ${
                        selectedImage === image.id ? "ring-2 ring-rainbow-blue" : ""
                      }`}
                      onClick={() => setSelectedImage(image.id === selectedImage ? null : image.id)}
                    >
                      <div className="relative aspect-square">
                        <Image src={image.url || "/placeholder.svg"} alt={image.name} fill className="object-cover" />
                      </div>
                      <CardContent className="p-2">
                        <p className="text-xs truncate">{image.name}</p>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </div>
            </TabsContent>
          </Tabs>
        </div>

        <div>
          <Card>
            <CardContent className="p-4">
              <h2 className="text-lg font-semibold mb-4">Lataa uusi kuva</h2>
              <form onSubmit={handleUpload} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="image">Valitse kuva</Label>
                  <Input id="image" type="file" accept="image/*" />
                </div>
                <Button type="submit" disabled={isUploading} className="w-full">
                  {isUploading ? (
                    <>Ladataan...</>
                  ) : (
                    <>
                      <Upload className="mr-2 h-4 w-4" />
                      Lataa kuva
                    </>
                  )}
                </Button>
              </form>
            </CardContent>
          </Card>

          {selectedImage && (
            <Card className="mt-6">
              <CardContent className="p-4">
                <h2 className="text-lg font-semibold mb-4">Kuvan tiedot</h2>
                {mockImages
                  .filter((img) => img.id === selectedImage)
                  .map((image) => (
                    <div key={image.id} className="space-y-4">
                      <div className="relative aspect-video rounded-md overflow-hidden">
                        <Image src={image.url || "/placeholder.svg"} alt={image.name} fill className="object-cover" />
                      </div>
                      <div className="space-y-2">
                        <div>
                          <span className="text-sm font-medium text-gray-500">Nimi:</span>
                          <span className="ml-2">{image.name}</span>
                        </div>
                        <div>
                          <span className="text-sm font-medium text-gray-500">Tyyppi:</span>
                          <span className="ml-2">{image.type}</span>
                        </div>
                        <div>
                          <span className="text-sm font-medium text-gray-500">Koko:</span>
                          <span className="ml-2">{image.size}</span>
                        </div>
                        <div>
                          <span className="text-sm font-medium text-gray-500">URL:</span>
                          <span className="ml-2 text-xs break-all">{image.url}</span>
                        </div>
                      </div>
                      <div className="pt-2">
                        <Button variant="destructive" className="w-full">
                          <Trash2 className="mr-2 h-4 w-4" />
                          Poista kuva
                        </Button>
                      </div>
                    </div>
                  ))}
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  )
}
