"use client"

import { useEffect, useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Users, FileText, Calendar, Clock } from "lucide-react"
import { ClientAdminLayout } from "@/components/admin/client-layout"
import dataService, { type Registration } from "@/app/lib/data-service"
import Link from "next/link"

export default function DashboardPage() {
  const [registrations, setRegistrations] = useState<Registration[]>([])
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // Initialize sample data if needed
    dataService.initializeWithSampleData()

    // Load registrations
    const data = dataService.getRegistrations()
    setRegistrations(data)
    setIsLoading(false)
  }, [])

  // Calculate some stats
  const pendingRegistrations = registrations.filter((r) => r.status === "pending").length
  const totalRegistrations = registrations.length

  if (isLoading) {
    return (
      <ClientAdminLayout>
        <div className="p-6">Loading...</div>
      </ClientAdminLayout>
    )
  }

  return (
    <ClientAdminLayout>
      <div className="p-6">
        <h1 className="text-2xl font-bold mb-6">Hallintapaneeli</h1>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium">Rekisteröinnit</CardTitle>
              <Users className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{totalRegistrations}</div>
              <p className="text-xs text-muted-foreground">{pendingRegistrations} käsittelemätöntä</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium">Sisältösivut</CardTitle>
              <FileText className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">5</div>
              <p className="text-xs text-muted-foreground">Etusivu, Tietoa meistä, Palvelut...</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium">Päivämäärä</CardTitle>
              <Calendar className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{new Date().toLocaleDateString("fi-FI")}</div>
              <p className="text-xs text-muted-foreground">
                {new Date().toLocaleDateString("fi-FI", { weekday: "long" })}
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between pb-2">
              <CardTitle className="text-sm font-medium">Aukioloaika</CardTitle>
              <Clock className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">7:00 - 17:00</div>
              <p className="text-xs text-muted-foreground">Ma-Pe (Suljettu La-Su)</p>
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Card>
            <CardHeader>
              <CardTitle>Viimeisimmät rekisteröinnit</CardTitle>
            </CardHeader>
            <CardContent>
              {registrations.length > 0 ? (
                <div className="space-y-4">
                  {registrations.slice(0, 5).map((registration) => (
                    <div key={registration.id} className="flex items-center justify-between border-b pb-2">
                      <div>
                        <p className="font-medium">{registration.childName}</p>
                        <p className="text-sm text-muted-foreground">{registration.guardianName}</p>
                      </div>
                      <div>
                        <span
                          className={`px-2 py-1 rounded-full text-xs ${
                            registration.status === "pending"
                              ? "bg-yellow-100 text-yellow-800"
                              : "bg-green-100 text-green-800"
                          }`}
                        >
                          {registration.status === "pending" ? "Odottaa" : "Käsitelty"}
                        </span>
                      </div>
                    </div>
                  ))}
                  <div className="pt-2">
                    <Link href="/admin/registrations" className="text-sm text-blue-600 hover:underline">
                      Näytä kaikki rekisteröinnit →
                    </Link>
                  </div>
                </div>
              ) : (
                <p className="text-muted-foreground">Ei rekisteröintejä vielä.</p>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Pikaohjeet</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <h3 className="font-medium">Rekisteröintien hallinta</h3>
                  <p className="text-sm text-muted-foreground">
                    Voit tarkastella ja hallita rekisteröintejä "Rekisteröinnit"-sivulla.
                  </p>
                </div>

                <div>
                  <h3 className="font-medium">Sisällön muokkaaminen</h3>
                  <p className="text-sm text-muted-foreground">
                    Muokkaa sivujen sisältöä "Sisältö"-sivulla. Voit muokata tekstejä, kuvia ja muita elementtejä.
                  </p>
                </div>

                <div>
                  <h3 className="font-medium">Median hallinta</h3>
                  <p className="text-sm text-muted-foreground">
                    Lataa ja hallitse kuvia "Media"-sivulla. Voit käyttää näitä kuvia sivuston eri osissa.
                  </p>
                </div>

                <div>
                  <h3 className="font-medium">Asetukset</h3>
                  <p className="text-sm text-muted-foreground">
                    Muuta sivuston yleisiä asetuksia, kuten logoa ja yhteystietoja "Asetukset"-sivulla.
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </ClientAdminLayout>
  )
}
