"use client"

import { useEffect, useState } from "react"
import { Button } from "@/components/ui/button"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Textarea } from "@/components/ui/textarea"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { MapPin } from "lucide-react"
import Link from "next/link"
import { ClientAdminLayout } from "@/components/admin/client-layout"
import dataService, { type SiteContent } from "@/app/lib/data-service"

export default function ContentPage() {
  const [activeTab, setActiveTab] = useState("etusivu")
  const [isSaving, setIsSaving] = useState(false)
  const [saveMessage, setSaveMessage] = useState<string | null>(null)
  const [content, setContent] = useState<SiteContent | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // Initialize sample data if needed
    dataService.initializeWithSampleData()

    // Load content
    const data = dataService.getSiteContent()
    setContent(data)
    setIsLoading(false)
  }, [])

  const handleSave = () => {
    if (!content) return

    setIsSaving(true)
    setSaveMessage(null)

    try {
      // Save content using our data service
      dataService.updateSiteContent(content)

      setSaveMessage("Sisältö tallennettu onnistuneesti!")

      // Clear message after 3 seconds
      setTimeout(() => {
        setSaveMessage(null)
      }, 3000)
    } catch (error) {
      console.error("Error saving content:", error)
      setSaveMessage("Virhe tallennettaessa sisältöä")
    } finally {
      setIsSaving(false)
    }
  }

  const handleChange = (section: keyof SiteContent, field: string, value: any) => {
    if (!content) return

    setContent({
      ...content,
      [section]: {
        ...content[section],
        [field]: value,
      },
    })
  }

  const handleServicesListChange = (value: string) => {
    if (!content) return

    setContent({
      ...content,
      services: {
        ...content.services,
        servicesList: value.split("\n").filter((item) => item.trim() !== ""),
      },
    })
  }

  if (isLoading || !content) {
    return (
      <ClientAdminLayout>
        <div className="p-6">Loading...</div>
      </ClientAdminLayout>
    )
  }

  return (
    <ClientAdminLayout>
      <div className="p-6">
        <div className="flex justify-between items-center mb-6">
          <h1 className="text-2xl font-bold">Sisällönhallinta</h1>
          <Button onClick={handleSave} disabled={isSaving}>
            {isSaving ? "Tallennetaan..." : "Tallenna muutokset"}
          </Button>
        </div>

        {saveMessage && (
          <div className="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
            {saveMessage}
          </div>
        )}

        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList className="mb-6">
            <TabsTrigger value="etusivu">Etusivu</TabsTrigger>
            <TabsTrigger value="tietoa-meista">Tietoa meistä</TabsTrigger>
            <TabsTrigger value="palvelut">Palvelut</TabsTrigger>
            <TabsTrigger value="yhteystiedot">Yhteystiedot</TabsTrigger>
          </TabsList>

          <TabsContent value="etusivu">
            <div className="space-y-4">
              <div>
                <Label htmlFor="hero-title">Otsikko</Label>
                <Input
                  id="hero-title"
                  value={content.hero.title}
                  onChange={(e) => handleChange("hero", "title", e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="hero-subtitle">Alaotsikko</Label>
                <Input
                  id="hero-subtitle"
                  value={content.hero.subtitle}
                  onChange={(e) => handleChange("hero", "subtitle", e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="hero-text">Kuvaus</Label>
                <Textarea
                  id="hero-text"
                  value={content.hero.description}
                  onChange={(e) => handleChange("hero", "description", e.target.value)}
                  rows={4}
                />
              </div>

              <div className="pt-4">
                <Button variant="outline" asChild>
                  <Link href="/" target="_blank">
                    Esikatsele etusivu
                  </Link>
                </Button>
              </div>
            </div>
          </TabsContent>

          <TabsContent value="tietoa-meista">
            <div className="space-y-4">
              <div>
                <Label htmlFor="about-main">Pääteksti</Label>
                <Textarea
                  id="about-main"
                  value={content.about.mainText}
                  onChange={(e) => handleChange("about", "mainText", e.target.value)}
                  rows={4}
                />
              </div>

              <div>
                <Label htmlFor="about-mission">Toiminta-ajatus</Label>
                <Textarea
                  id="about-mission"
                  value={content.about.mission}
                  onChange={(e) => handleChange("about", "mission", e.target.value)}
                  rows={3}
                />
              </div>

              <div>
                <Label htmlFor="about-vision">Visio</Label>
                <Textarea
                  id="about-vision"
                  value={content.about.vision}
                  onChange={(e) => handleChange("about", "vision", e.target.value)}
                  rows={3}
                />
              </div>

              <div className="pt-4">
                <Button variant="outline" asChild>
                  <Link href="/tietoa-meista" target="_blank">
                    Esikatsele Tietoa meistä
                  </Link>
                </Button>
              </div>
            </div>
          </TabsContent>

          <TabsContent value="palvelut">
            <div className="space-y-4">
              <div>
                <Label htmlFor="services-main">Pääteksti</Label>
                <Textarea
                  id="services-main"
                  value={content.services.mainText}
                  onChange={(e) => handleChange("services", "mainText", e.target.value)}
                  rows={4}
                />
              </div>

              <div>
                <Label htmlFor="services-list">Palvelut (yksi per rivi)</Label>
                <Textarea
                  id="services-list"
                  value={content.services.servicesList.join("\n")}
                  onChange={(e) => handleServicesListChange(e.target.value)}
                  rows={6}
                />
              </div>

              <div className="pt-4">
                <Button variant="outline" asChild>
                  <Link href="/palvelut" target="_blank">
                    Esikatsele Palvelut
                  </Link>
                </Button>
              </div>
            </div>
          </TabsContent>

          <TabsContent value="yhteystiedot">
            <div className="space-y-4">
              <div>
                <Label htmlFor="contact-address">Osoite</Label>
                <Input
                  id="contact-address"
                  value={content.contact.address}
                  onChange={(e) => handleChange("contact", "address", e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="contact-map">Google Maps -linkki</Label>
                <div className="flex gap-2">
                  <Input
                    id="contact-map"
                    value={content.contact.mapUrl}
                    onChange={(e) => handleChange("contact", "mapUrl", e.target.value)}
                    placeholder="https://maps.app.goo.gl/..."
                  />
                  <Link href={content.contact.mapUrl} target="_blank" rel="noopener noreferrer">
                    <Button type="button" variant="outline" size="icon">
                      <MapPin className="h-4 w-4" />
                    </Button>
                  </Link>
                </div>
              </div>

              <div>
                <Label htmlFor="contact-phone">Puhelinnumero</Label>
                <Input
                  id="contact-phone"
                  value={content.contact.phone}
                  onChange={(e) => handleChange("contact", "phone", e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="contact-email">Sähköposti</Label>
                <Input
                  id="contact-email"
                  value={content.contact.email}
                  onChange={(e) => handleChange("contact", "email", e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="contact-hours">Aukioloajat</Label>
                <Input
                  id="contact-hours"
                  value={content.contact.hours}
                  onChange={(e) => handleChange("contact", "hours", e.target.value)}
                />
              </div>

              <div className="pt-4">
                <Button variant="outline" asChild>
                  <Link href="/yhteystiedot" target="_blank">
                    Esikatsele Yhteystiedot
                  </Link>
                </Button>
              </div>
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </ClientAdminLayout>
  )
}
