<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SiteContent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ContentController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('admin');
    }
    
    public function index()
    {
        $contents = SiteContent::orderBy('section')->orderBy('order')->paginate(15);
        return view('admin.content.index', compact('contents'));
    }
    
    public function create()
    {
        $sections = ['hero', 'about', 'services', 'gallery', 'contact'];
        return view('admin.content.create', compact('sections'));
    }
    
    public function store(Request $request)
    {
        $validated = $request->validate([
            'section' => 'required|string|max:50',
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'order' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);
        
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('public/content');
            $validated['image_path'] = Storage::url($path);
        }
        
        $validated['is_active'] = $request->has('is_active');
        
        SiteContent::create($validated);
        
        return redirect()->route('admin.content.index')
            ->with('success', 'Content created successfully.');
    }
    
    public function edit(SiteContent $content)
    {
        $sections = ['hero', 'about', 'services', 'gallery', 'contact'];
        return view('admin.content.edit', compact('content', 'sections'));
    }
    
    public function update(Request $request, SiteContent $content)
    {
        $validated = $request->validate([
            'section' => 'required|string|max:50',
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'order' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);
        
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($content->image_path) {
                $oldPath = str_replace('/storage', 'public', $content->image_path);
                Storage::delete($oldPath);
            }
            
            $path = $request->file('image')->store('public/content');
            $validated['image_path'] = Storage::url($path);
        }
        
        $validated['is_active'] = $request->has('is_active');
        
        $content->update($validated);
        
        return redirect()->route('admin.content.index')
            ->with('success', 'Content updated successfully.');
    }
    
    public function destroy(SiteContent $content)
    {
        // Delete image if exists
        if ($content->image_path) {
            $path = str_replace('/storage', 'public', $content->image_path);
            Storage::delete($path);
        }
        
        $content->delete();
        
        return redirect()->route('admin.content.index')
            ->with('success', 'Content deleted successfully.');
    }
}
